/*
 * Decompiled with CFR 0.152.
 */
package org.esa.beam.dataio.ceos.prism;

import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import javax.imageio.stream.FileImageInputStream;
import javax.imageio.stream.ImageInputStream;
import org.esa.beam.dataio.ceos.CeosHelper;
import org.esa.beam.dataio.ceos.IllegalCeosFormatException;
import org.esa.beam.dataio.ceos.prism.PrismImageFile;
import org.esa.beam.dataio.ceos.prism.PrismLeaderFile;
import org.esa.beam.dataio.ceos.prism.PrismSupplementalFile;
import org.esa.beam.dataio.ceos.prism.PrismTrailerFile;
import org.esa.beam.dataio.ceos.prism.PrismVolumeDirectoryFile;
import org.esa.beam.dataio.ceos.prism.records.PrismAncillary2Record;
import org.esa.beam.dataio.ceos.prism.records.SceneHeaderRecord;
import org.esa.beam.dataio.ceos.records.Ancillary1Record;
import org.esa.beam.dataio.ceos.records.Ancillary3Record;
import org.esa.beam.framework.datamodel.Band;
import org.esa.beam.framework.datamodel.CombinedFXYGeoCoding;
import org.esa.beam.framework.datamodel.FXYGeoCoding;
import org.esa.beam.framework.datamodel.GeoCoding;
import org.esa.beam.framework.datamodel.GeoPos;
import org.esa.beam.framework.datamodel.MapGeoCoding;
import org.esa.beam.framework.datamodel.MetadataAttribute;
import org.esa.beam.framework.datamodel.MetadataElement;
import org.esa.beam.framework.datamodel.Product;
import org.esa.beam.framework.datamodel.ProductData;
import org.esa.beam.framework.dataop.maptransf.Datum;
import org.esa.beam.framework.dataop.maptransf.Ellipsoid;
import org.esa.beam.framework.dataop.maptransf.MapInfo;
import org.esa.beam.framework.dataop.maptransf.MapProjection;
import org.esa.beam.framework.dataop.maptransf.MapTransform;
import org.esa.beam.framework.dataop.maptransf.MapTransformFactory;
import org.esa.beam.framework.dataop.maptransf.StereographicDescriptor;
import org.esa.beam.framework.dataop.maptransf.UTM;
import org.esa.beam.util.Debug;
import org.esa.beam.util.Guardian;
import org.esa.beam.util.StringUtils;
import org.esa.beam.util.TreeNode;
import org.esa.beam.util.math.FXYSum;

class PrismProductDirectory {
    private static final int METER_PER_KILOMETER = 1000;
    private static final String MAP_PROJECTION_CODE_RAW = "NNNNN";
    private static final String MAP_PROJECTION_CODE_UTM = "YNNNN";
    private static final String MAP_PROJECTION_CODE_PS = "NNNNY";
    private final File _baseDir;
    private PrismVolumeDirectoryFile _volumeDirectoryFile;
    private PrismImageFile[] _imageFiles;
    private PrismLeaderFile _leaderFile;
    private PrismTrailerFile _trailerFile;
    private PrismSupplementalFile _supplementalFile;
    private static final String UNIT_METER = "meter";
    private static final String UNIT_KILOMETER = "kilometer";
    private static final String UNIT_DEGREE = "degree";
    private static final String UNIT_SECOND = "second";
    private static final String UNIT_METER_PER_SECOND = "m/sec";
    private static final String UNIT_DEGREE_PER_SECOND = "deg/sec";

    PrismProductDirectory(File dir) throws IOException, IllegalCeosFormatException {
        Guardian.assertNotNull((String)"dir", (Object)dir);
        this._baseDir = dir;
        this._volumeDirectoryFile = new PrismVolumeDirectoryFile(this._baseDir);
        this._leaderFile = new PrismLeaderFile(this.createInputStream(this._volumeDirectoryFile.getLeaderFileName()));
        this._trailerFile = new PrismTrailerFile(this.createInputStream(this._volumeDirectoryFile.getTrailerFileName()));
        if (!this._leaderFile.getProductLevel().equalsIgnoreCase("1B2")) {
            this._supplementalFile = new PrismSupplementalFile(this.createInputStream(this._volumeDirectoryFile.getSupplementalFileName()));
        }
        String[] imageFileNames = this._volumeDirectoryFile.getImageFileNames();
        this._imageFiles = new PrismImageFile[imageFileNames.length];
        for (int i = 0; i < imageFileNames.length; ++i) {
            this._imageFiles[i] = new PrismImageFile(this.createInputStream(imageFileNames[i]));
        }
    }

    Product createProduct() throws IOException, IllegalCeosFormatException {
        String productName = this._leaderFile.getProductName();
        String productType = this._leaderFile.getProductType();
        int width = 0;
        int overlap = this._imageFiles.length > 1 ? 32 : 0;
        for (int i = 0; i < this._imageFiles.length; ++i) {
            PrismImageFile imageFile = this._imageFiles[i];
            width += imageFile.getWidth() - overlap;
        }
        int sceneWidth = width;
        int sceneHeight = this._leaderFile.getSceneHeight();
        Product product = new Product(productName, productType, sceneWidth, sceneHeight);
        product.setFileLocation(this._baseDir);
        this.addBand(product);
        Calendar imageStartDate = this._leaderFile.getDateImageWasTaken();
        imageStartDate.add(14, this._imageFiles[0].getTotalMillisInDayOfLine(0));
        ProductData.UTC utcScanStartTime = ProductData.UTC.create((Date)imageStartDate.getTime(), (long)this._imageFiles[0].getMicrosecondsOfLine(0));
        Calendar imageEndDate = this._leaderFile.getDateImageWasTaken();
        imageEndDate.add(14, this._imageFiles[0].getTotalMillisInDayOfLine(sceneHeight - 1));
        ProductData.UTC utcScanStopTime = ProductData.UTC.create((Date)imageEndDate.getTime(), (long)this._imageFiles[0].getMicrosecondsOfLine(sceneHeight - 1));
        product.setStartTime(utcScanStartTime);
        product.setEndTime(utcScanStopTime);
        product.setDescription("PRISM product Level " + productType);
        this.addGeoCoding(product);
        this.addMetadataTo(product);
        return product;
    }

    private void addBand(Product product) {
        String bandName = "radiance_1";
        Band band = new Band("radiance_1", 20, product.getSceneRasterWidth(), product.getSceneRasterHeight());
        band.setSpectralBandIndex(0);
        band.setSpectralWavelength(645.0f);
        band.setSpectralBandwidth(250.0f);
        band.setUnit("mw / (m^2*sr*nm)");
        band.setScalingFactor(this._leaderFile.getAncillary2Record().getAbsoluteCalibrationGain());
        band.setScalingOffset(this._leaderFile.getAncillary2Record().getAbsoluteCalibrationOffset());
        band.setNoDataValue(0.0);
        band.setNoDataValueUsed(true);
        product.addBand(band);
    }

    void close() throws IOException {
        for (int i = 0; i < this._imageFiles.length; ++i) {
            this._imageFiles[i].close();
        }
        this._leaderFile.close();
        if (this._supplementalFile != null) {
            this._supplementalFile.close();
        }
        this._trailerFile.close();
        this._volumeDirectoryFile.close();
    }

    PrismImageFile[] getImageFiles() {
        return this._imageFiles;
    }

    TreeNode<File> getProductComponents() {
        try {
            File parentDir = this._baseDir.getParentFile();
            TreeNode root = new TreeNode(parentDir.getCanonicalPath());
            root.setContent((Object)parentDir);
            TreeNode dir = new TreeNode(this._baseDir.getName());
            dir.setContent((Object)this._baseDir);
            root.addChild(dir);
            File volumeFile = CeosHelper.getVolumeFile(this._baseDir);
            this.addFileToDir((TreeNode<File>)dir, volumeFile);
            this.addFileToDir((TreeNode<File>)dir, new File(this._volumeDirectoryFile.getLeaderFileName()));
            this.addFileToDir((TreeNode<File>)dir, new File(this._volumeDirectoryFile.getTrailerFileName()));
            String supplemental = this._volumeDirectoryFile.getSupplementalFileName();
            if (StringUtils.isNotNullAndNotEmpty((String)supplemental)) {
                this.addFileToDir((TreeNode<File>)dir, new File(supplemental));
            }
            String[] imageFileNames = this._volumeDirectoryFile.getImageFileNames();
            for (int i = 0; i < imageFileNames.length; ++i) {
                this.addFileToDir((TreeNode<File>)dir, new File(imageFileNames[i]));
            }
            return root;
        }
        catch (IOException e) {
            return null;
        }
    }

    private void addFileToDir(TreeNode<File> dir, File file) {
        TreeNode fileNode = new TreeNode(file.getName());
        fileNode.setContent((Object)file);
        dir.addChild(fileNode);
    }

    private void addGeoCoding(Product product) throws IllegalCeosFormatException, IOException {
        String projectionCode = this.getProjectionCode();
        if (MAP_PROJECTION_CODE_RAW.equalsIgnoreCase(projectionCode)) {
            int overlap = 32;
            int pixelOffsetX = 16;
            PrismImageFile[] imageFiles = this.getImageFiles();
            CombinedFXYGeoCoding.CodingWrapper[] codingWrappers = new CombinedFXYGeoCoding.CodingWrapper[imageFiles.length];
            for (int i = 0; i < imageFiles.length; ++i) {
                PrismImageFile imageFile = imageFiles[i];
                int width = imageFile.getWidth() - 32;
                int height = imageFile.getHeight();
                int offset = width * i;
                int ccdNumber = imageFile.getImageNumber();
                double[][] uncorrectedCoeffs = this._leaderFile.getUncorrectedTransformationCoeffs(ccdNumber);
                FXYSum.Cubic funcLat = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[0]));
                FXYSum.Cubic funcLon = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[1]));
                FXYSum.Cubic funcX = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[2]));
                FXYSum.Cubic funcY = new FXYSum.Cubic(CeosHelper.sortToFXYSumOrder(uncorrectedCoeffs[3]));
                FXYGeoCoding gc = new FXYGeoCoding(16.0f, 0.0f, 1.0f, 1.0f, (FXYSum)funcX, (FXYSum)funcY, (FXYSum)funcLat, (FXYSum)funcLon, Datum.ITRF_97);
                if (i != imageFiles.length - 1) {
                    ++width;
                }
                codingWrappers[i] = new CombinedFXYGeoCoding.CodingWrapper(gc, offset, 0, width, height);
            }
            CombinedFXYGeoCoding gc = new CombinedFXYGeoCoding(codingWrappers);
            product.setGeoCoding((GeoCoding)gc);
        } else if (MAP_PROJECTION_CODE_UTM.equalsIgnoreCase(projectionCode)) {
            int meterPerKilometer = 1000;
            int sceneWidth = product.getSceneRasterWidth();
            int sceneHeight = product.getSceneRasterHeight();
            int zoneIndex = (int)this._leaderFile.getUTMZoneIndex();
            boolean isSouth = this._leaderFile.isUTMSouthHemisphere();
            double easting = this._leaderFile.getUTMEasting() * 1000.0;
            double northing = this._leaderFile.getUTMNorthing() * 1000.0;
            double pixelSizeX = this._leaderFile.getNominalInterPixelDistance();
            double pixelSizeY = this._leaderFile.getNominalInterLineDistance();
            float orientationAngle = (float)this._leaderFile.getUTMOrientationAngle();
            MapInfo mapInfo = new MapInfo((MapProjection)UTM.createProjection((int)(zoneIndex - 1), (boolean)isSouth), (float)sceneWidth * 0.5f, (float)sceneHeight * 0.5f, (float)easting, (float)northing, (float)pixelSizeX, (float)pixelSizeY, Datum.ITRF_97);
            mapInfo.setOrientation(-orientationAngle);
            mapInfo.setSceneWidth(sceneWidth);
            mapInfo.setSceneHeight(sceneHeight);
            product.setGeoCoding((GeoCoding)new MapGeoCoding(mapInfo));
        } else if (MAP_PROJECTION_CODE_PS.equalsIgnoreCase(projectionCode)) {
            double[] parameterValues = StereographicDescriptor.PARAMETER_DEFAULT_VALUES;
            parameterValues[0] = Ellipsoid.GRS_80.getSemiMajor();
            parameterValues[1] = Ellipsoid.GRS_80.getSemiMinor();
            GeoPos psReferencePoint = this._leaderFile.getPSReferencePoint();
            GeoPos psProjectionOrigin = this._leaderFile.getPSProjectionOrigin();
            parameterValues[2] = psProjectionOrigin.getLat();
            parameterValues[3] = psReferencePoint.getLon();
            MapTransform transform = MapTransformFactory.createTransform((String)"Stereographic", (double[])parameterValues);
            MapProjection projection = new MapProjection("Stereographic", transform);
            double pixelSizeX = this._leaderFile.getNominalInterPixelDistance();
            double pixelSizeY = this._leaderFile.getNominalInterLineDistance();
            double easting = this._leaderFile.getPSXCoordinate() * 1000.0;
            double northing = this._leaderFile.getPSYCoordinate() * 1000.0;
            int sceneRasterWidth = product.getSceneRasterWidth();
            int sceneRasterHeight = product.getSceneRasterHeight();
            MapInfo mapInfo = new MapInfo(projection, (float)sceneRasterWidth * 0.5f, (float)sceneRasterHeight * 0.5f, (float)easting, (float)northing, (float)pixelSizeX, (float)pixelSizeY, Datum.ITRF_97);
            mapInfo.setOrientation((float)this._leaderFile.getPSOrientationAngle());
            mapInfo.setSceneWidth(sceneRasterWidth);
            mapInfo.setSceneHeight(sceneRasterHeight);
            product.setGeoCoding((GeoCoding)new MapGeoCoding(mapInfo));
        } else {
            Debug.trace((String)"Unknown map projection method. Could not create geo-coding.");
        }
    }

    private void addMetadataTo(Product product) throws IOException, IllegalCeosFormatException {
        MetadataElement metadataRoot = product.getMetadataRoot();
        this.assignSPH(metadataRoot);
        this._volumeDirectoryFile.assignMetadataTo(metadataRoot);
    }

    private void assignSPH(MetadataElement metadataRoot) throws IOException, IllegalCeosFormatException {
        MetadataElement sphElement = new MetadataElement("SPH");
        metadataRoot.addElement(sphElement);
        sphElement.addElement(this.getMapProjectionMetadata());
        sphElement.addElement(this.getRadiometricMetadata());
        sphElement.addElement(this.getPlatformMetadata());
    }

    private MetadataElement getMapProjectionMetadata() throws IOException, IllegalCeosFormatException {
        MetadataElement projMetadata = new MetadataElement("Map Projection");
        this.addGeneralProjectionMetadata(projMetadata);
        String usedProjection = this.getProjectionCode();
        if (usedProjection.equalsIgnoreCase(MAP_PROJECTION_CODE_RAW)) {
            this.addRawProjectionMetadata(projMetadata);
        } else if (usedProjection.equalsIgnoreCase(MAP_PROJECTION_CODE_UTM)) {
            this.addGeneralCorrectedMetadata(projMetadata);
            this.addUTMProjectionMetadata(projMetadata);
        } else if (usedProjection.equalsIgnoreCase(MAP_PROJECTION_CODE_PS)) {
            this.addGeneralCorrectedMetadata(projMetadata);
            this.addPSProjectionMetadata(projMetadata);
        }
        return projMetadata;
    }

    private void addGeneralProjectionMetadata(MetadataElement projMeta) throws IOException, IllegalCeosFormatException {
        PrismLeaderFile lf = this._leaderFile;
        Ancillary1Record a1r = lf.getAncillary1Record();
        PrismProductDirectory.addAttribute(projMeta, "REFERENCE_ELLIPSOID", ProductData.createInstance((String)a1r.getReferenceEllipsoid()));
        PrismProductDirectory.addAttribute(projMeta, "SEMI_MAJOR_AXIS", ProductData.createInstance((double[])new double[]{lf.getSemiMajorAxis()}), UNIT_METER);
        PrismProductDirectory.addAttribute(projMeta, "SEMI_MINOR_AXIS", ProductData.createInstance((double[])new double[]{lf.getSemiMinorAxis()}), UNIT_METER);
        PrismProductDirectory.addAttribute(projMeta, "GEODETIC_DATUM", ProductData.createInstance((String)lf.getDatumName()));
        double[] latCorners = this.getLatCorners();
        double[] lonCorners = this.getLonCorners();
        PrismProductDirectory.addAttribute(projMeta, "SCENE_UPPER_LEFT_LATITUDE", ProductData.createInstance((double[])new double[]{latCorners[0]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_UPPER_LEFT_LONGITUDE", ProductData.createInstance((double[])new double[]{lonCorners[0]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_UPPER_RIGHT_LATITUDE", ProductData.createInstance((double[])new double[]{latCorners[1]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_UPPER_RIGHT_LONGITUDE", ProductData.createInstance((double[])new double[]{lonCorners[1]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_LOWER_LEFT_LATITUDE", ProductData.createInstance((double[])new double[]{latCorners[2]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_LOWER_LEFT_LONGITUDE", ProductData.createInstance((double[])new double[]{lonCorners[2]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_LOWER_RIGHT_LATITUDE", ProductData.createInstance((double[])new double[]{latCorners[3]}), UNIT_DEGREE);
        PrismProductDirectory.addAttribute(projMeta, "SCENE_LOWER_RIGHT_LONGITUDE", ProductData.createInstance((double[])new double[]{lonCorners[3]}), UNIT_DEGREE);
    }

    private void addRawProjectionMetadata(MetadataElement projMeta) throws IOException, IllegalCeosFormatException {
        PrismLeaderFile lf = this._leaderFile;
        for (int i = 1; i <= 4; ++i) {
            int j;
            double[][] uncorrectedTransformationCoeffs = lf.getUncorrectedTransformationCoeffs(i);
            for (j = 0; j < uncorrectedTransformationCoeffs[0].length; ++j) {
                double coeffLat = uncorrectedTransformationCoeffs[0][j];
                PrismProductDirectory.addAttribute(projMeta, "CCD[" + i + "]_COEFFICIENTS_LATITUDE." + j, ProductData.createInstance((double[])new double[]{coeffLat}));
            }
            for (j = 0; j < uncorrectedTransformationCoeffs[1].length; ++j) {
                double coeffLon = uncorrectedTransformationCoeffs[1][j];
                PrismProductDirectory.addAttribute(projMeta, "CCD[" + i + "]_COEFFICIENTS_LONGITUDE." + j, ProductData.createInstance((double[])new double[]{coeffLon}));
            }
            for (j = 0; j < uncorrectedTransformationCoeffs[2].length; ++j) {
                double coeffX = uncorrectedTransformationCoeffs[2][j];
                PrismProductDirectory.addAttribute(projMeta, "CCD[" + i + "]_COEFFICIENTS_X." + j, ProductData.createInstance((double[])new double[]{coeffX}));
            }
            for (j = 0; j < uncorrectedTransformationCoeffs[3].length; ++j) {
                double coeffY = uncorrectedTransformationCoeffs[3][j];
                PrismProductDirectory.addAttribute(projMeta, "CCD[" + i + "]_COEFFICIENTS_Y." + j, ProductData.createInstance((double[])new double[]{coeffY}));
            }
        }
        PrismProductDirectory.addAttribute(projMeta, "PIXELS_PER_LINE", ProductData.createInstance((long[])new long[]{lf.getNominalPixelsPerLine_1A_1B1()}));
        PrismProductDirectory.addAttribute(projMeta, "LINES_PER_SCENE", ProductData.createInstance((long[])new long[]{lf.getNominalLinesPerScene_1A_1B1()}));
        PrismProductDirectory.addAttribute(projMeta, "PIXEL_SIZE_X_CENTER", ProductData.createInstance((double[])new double[]{lf.getNominalInterPixelDistance_1A_1B1()}), UNIT_METER);
        PrismProductDirectory.addAttribute(projMeta, "PIXEL_SIZE_Y_CENTER", ProductData.createInstance((double[])new double[]{lf.getNominalInterLineDistance_1A_1B1()}), UNIT_METER);
        PrismProductDirectory.addAttribute(projMeta, "IMAGE_SKEW_CENTER", ProductData.createInstance((double[])new double[]{lf.getImageSkew()}), "milliradian");
    }

    private void addUTMProjectionMetadata(MetadataElement projMeta) throws IOException, IllegalCeosFormatException {
        PrismLeaderFile lf = this._leaderFile;
        PrismProductDirectory.addAttribute(projMeta, "HEMISPHERE", ProductData.createInstance((String)(lf.isUTMSouthHemisphere() ? "South" : "North")));
        PrismProductDirectory.addAttribute(projMeta, "UTM_ZONE_NUMBER", ProductData.createInstance((long[])new long[]{lf.getUTMZoneIndex()}));
        PrismProductDirectory.addAttribute(projMeta, "UTM_NORTHING", ProductData.createInstance((double[])new double[]{lf.getUTMNorthing()}), UNIT_KILOMETER);
        PrismProductDirectory.addAttribute(projMeta, "UTM_EASTING", ProductData.createInstance((double[])new double[]{lf.getUTMEasting()}), UNIT_KILOMETER);
        MetadataAttribute orientation = PrismProductDirectory.addAttribute(projMeta, "ORIENTATION", ProductData.createInstance((double[])new double[]{lf.getUTMOrientationAngle()}), UNIT_DEGREE);
        orientation.setDescription("Angle between the map projection vertical axis and the true north at scene center");
    }

    private void addPSProjectionMetadata(MetadataElement projMeta) throws IOException, IllegalCeosFormatException {
        PrismLeaderFile lf = this._leaderFile;
        GeoPos origin = lf.getPSProjectionOrigin();
        PrismProductDirectory.addAttribute(projMeta, "MAP_PROJECTION_ORIGIN", ProductData.createInstance((String)(origin.getLatString() + " , " + origin.getLonString())));
        GeoPos reference = lf.getPSReferencePoint();
        PrismProductDirectory.addAttribute(projMeta, "REFERENCE_POINT", ProductData.createInstance((String)(reference.getLatString() + " , " + reference.getLonString())));
        PrismProductDirectory.addAttribute(projMeta, "COORDINATE_CENTER_X", ProductData.createInstance((double[])new double[]{lf.getPSXCoordinate()}), UNIT_KILOMETER);
        PrismProductDirectory.addAttribute(projMeta, "COORDINATE_CENTER_Y)", ProductData.createInstance((double[])new double[]{lf.getPSYCoordinate()}), UNIT_KILOMETER);
        MetadataAttribute orientation = PrismProductDirectory.addAttribute(projMeta, "ORIENTATION", ProductData.createInstance((double[])new double[]{lf.getPSOrientationAngle()}), UNIT_DEGREE);
        orientation.setDescription("Angle between the map projection vertical axis and the true north at scene center");
    }

    private void addGeneralCorrectedMetadata(MetadataElement projMeta) throws IllegalCeosFormatException, IOException {
        Ancillary1Record ar = this._leaderFile.getAncillary1Record();
        PrismProductDirectory.addAttribute(projMeta, "PIXELS_PER_LINE", ProductData.createInstance((double[])new double[]{ar.getNumNominalPixelsPerLine()}));
        PrismProductDirectory.addAttribute(projMeta, "LINES_PER_SCENE", ProductData.createInstance((double[])new double[]{ar.getNumNominalLinesPerScene()}));
        PrismProductDirectory.addAttribute(projMeta, "PIXEL_SIZE_X_CENTER", ProductData.createInstance((double[])new double[]{ar.getNominalInterPixelDistance()}), UNIT_METER);
        PrismProductDirectory.addAttribute(projMeta, "PIXEL_SIZE_Y_CENTER", ProductData.createInstance((double[])new double[]{ar.getNominalInterLineDistance()}), UNIT_METER);
    }

    private MetadataElement getRadiometricMetadata() {
        PrismAncillary2Record ar = this._leaderFile.getAncillary2Record();
        MetadataElement radioMetadata = new MetadataElement("Radiometric Calibration");
        PrismProductDirectory.addAttribute(radioMetadata, "SENSOR_MODE", ProductData.createInstance((String)ar.getSensorOperationMode()));
        PrismProductDirectory.addAttribute(radioMetadata, "LOWER_LIMIT_STRENGTH", ProductData.createInstance((int[])new int[]{ar.getLowerLimitOfStrengthAfterCorrection()}));
        PrismProductDirectory.addAttribute(radioMetadata, "UPPER_LIMIT_STRENGTH", ProductData.createInstance((int[])new int[]{ar.getLowerLimitOfStrengthAfterCorrection()}));
        char[] gains = ar.getSensorGains().trim().toCharArray();
        for (int i = 0; i < gains.length; ++i) {
            PrismProductDirectory.addAttribute(radioMetadata, "SENSOR_GAIN." + (i + 1), ProductData.createInstance((String)(gains[i] == ' ' ? "" : "Gain " + gains[i])));
        }
        PrismProductDirectory.addAttribute(radioMetadata, "SIGNAL_PROCESSING_UNIT_TEMPERATURE", ProductData.createInstance((double[])new double[]{ar.getSignalProcessingSectionTemperature()}), UNIT_DEGREE);
        double absGain = ar.getAbsoluteCalibrationGain();
        double absOffset = ar.getAbsoluteCalibrationOffset();
        PrismProductDirectory.addAttribute(radioMetadata, "ABSOLUTE_GAIN_BAND", ProductData.createInstance((double[])new double[]{absGain}));
        PrismProductDirectory.addAttribute(radioMetadata, "ABSOLUTE_OFFSET_BAND", ProductData.createInstance((double[])new double[]{absOffset}));
        return radioMetadata;
    }

    private MetadataElement getPlatformMetadata() {
        Ancillary3Record ar = this._leaderFile.getAncillary3Record();
        MetadataElement platformMeta = new MetadataElement("Platform Position Data");
        PrismProductDirectory.addAttribute(platformMeta, "NUMBER_EFFECTIVE_DATA_POINTS", ProductData.createInstance((int[])new int[]{ar.getNumDataPoints()}));
        PrismProductDirectory.addAttribute(platformMeta, "YEAR_OF_FIRST_POINT", ProductData.createInstance((int[])new int[]{ar.getFirstPointYear()}));
        PrismProductDirectory.addAttribute(platformMeta, "MONTH_OF_FIRST_POINT", ProductData.createInstance((int[])new int[]{ar.getFirstPointMonth()}));
        PrismProductDirectory.addAttribute(platformMeta, "DAY_OF_FIRST_POINT", ProductData.createInstance((int[])new int[]{ar.getFirstPointDay()}));
        PrismProductDirectory.addAttribute(platformMeta, "TOTAL_DAYS_OF_FIRST_POINT", ProductData.createInstance((int[])new int[]{ar.getFirstPointTotalDays()}));
        PrismProductDirectory.addAttribute(platformMeta, "TOTAL_SECONDS_OF_FIRST_POINT", ProductData.createInstance((double[])new double[]{ar.getFirstPointTotalSeconds()}));
        PrismProductDirectory.addAttribute(platformMeta, "POINTS_INTERVAL_TIME", ProductData.createInstance((double[])new double[]{ar.getIntervalTimeBetweenPoints()}), UNIT_SECOND);
        PrismProductDirectory.addAttribute(platformMeta, "REFERENCE_COORDINATE_SYSTEM", ProductData.createInstance((String)ar.getReferenceCoordinateSystem()));
        PrismProductDirectory.addAttribute(platformMeta, "POSITIONAL_ERROR_FLIGHT_DIRECTION", ProductData.createInstance((double[])new double[]{ar.getPositionalErrorFlightDirection()}), UNIT_METER);
        PrismProductDirectory.addAttribute(platformMeta, "POSITIONAL_ERROR_VERTICAL_FLIGHT_DIRECTION", ProductData.createInstance((double[])new double[]{ar.getPositionalErrorFlightVerticalDirection()}), UNIT_METER);
        PrismProductDirectory.addAttribute(platformMeta, "POSITIONAL_ERROR_RADIUS_DIRECTION", ProductData.createInstance((double[])new double[]{ar.getPositionalErrorRadiusDirection()}), UNIT_METER_PER_SECOND);
        PrismProductDirectory.addAttribute(platformMeta, "VELOCITY_ERROR_FLIGHT_DIRECTION", ProductData.createInstance((double[])new double[]{ar.getVelocityErrorFlightDirection()}), UNIT_METER_PER_SECOND);
        PrismProductDirectory.addAttribute(platformMeta, "VELOCITY_ERROR_VERTICAL_FLIGHT_DIRECTION", ProductData.createInstance((double[])new double[]{ar.getVelocityErrorFlightVerticalDirection()}), UNIT_METER_PER_SECOND);
        PrismProductDirectory.addAttribute(platformMeta, "VELOCITY_ERROR_RADIUS_DIRECTION", ProductData.createInstance((double[])new double[]{ar.getVelocityErrorRadiusDirection()}), UNIT_DEGREE_PER_SECOND);
        Ancillary3Record.DataPoint[] dataPoints = ar.getDataPoints();
        for (int i = 0; i < dataPoints.length; ++i) {
            int pIndex = i + 1;
            Ancillary3Record.DataPoint dataPoint = dataPoints[i];
            PrismProductDirectory.addAttribute(platformMeta, "DATA_POINT_" + pIndex + "_POSITIONAL_VECTOR_X", ProductData.createInstance((double[])new double[]{dataPoint.getPositionalVectorDataPointX()}));
            PrismProductDirectory.addAttribute(platformMeta, "DATA_POINT_" + pIndex + "_POSITIONAL_VECTOR_Y", ProductData.createInstance((double[])new double[]{dataPoint.getPositionalVectorDataPointY()}));
            PrismProductDirectory.addAttribute(platformMeta, "DATA_POINT_" + pIndex + "_POSITIONAL_VECTOR_Z", ProductData.createInstance((double[])new double[]{dataPoint.getPositionalVectorDataPointZ()}));
            PrismProductDirectory.addAttribute(platformMeta, "DATA_POINT_" + pIndex + "_VELOCITY_VECTOR_X", ProductData.createInstance((double[])new double[]{dataPoint.getVelocityVectorDataPointX()}));
            PrismProductDirectory.addAttribute(platformMeta, "DATA_POINT_" + pIndex + "_VELOCITY_VECTOR_Y", ProductData.createInstance((double[])new double[]{dataPoint.getVelocityVectorDataPointY()}));
            PrismProductDirectory.addAttribute(platformMeta, "DATA_POINT_" + pIndex + "_VELOCITY_VECTOR_Z", ProductData.createInstance((double[])new double[]{dataPoint.getVelocityVectorDataPointZ()}));
        }
        PrismProductDirectory.addAttribute(platformMeta, "LEAP_SECOND", ProductData.createInstance((String)String.valueOf(ar.getFlagLeapSecond() == 1)));
        return platformMeta;
    }

    private double[] getLatCorners() {
        SceneHeaderRecord shr = this._leaderFile.getSceneHeaderRecord();
        double latUL = shr.getSceneCornerUpperLeftLat();
        double latUR = shr.getSceneCornerUpperRightLat();
        double latLL = shr.getSceneCornerLowerLeftLat();
        double latLR = shr.getSceneCornerLowerRightLat();
        return new double[]{latUL, latUR, latLL, latLR};
    }

    private double[] getLonCorners() {
        SceneHeaderRecord shr = this._leaderFile.getSceneHeaderRecord();
        double lonUL = shr.getSceneCornerUpperLeftLon();
        double lonUR = shr.getSceneCornerUpperRightLon();
        double lonLL = shr.getSceneCornerLowerLeftLon();
        double lonLR = shr.getSceneCornerLowerLeftLat();
        return new double[]{lonUL, lonUR, lonLL, lonLR};
    }

    private String getProjectionCode() {
        return this._leaderFile.getSceneHeaderRecord().getMapProjectionMethod().trim();
    }

    private static MetadataAttribute createAttribute(String name, ProductData data) {
        return new MetadataAttribute(name.toUpperCase(), data, true);
    }

    private static MetadataAttribute addAttribute(MetadataElement platformMetadata, String name, ProductData data) {
        return PrismProductDirectory.addAttribute(platformMetadata, name, data, null);
    }

    private static MetadataAttribute addAttribute(MetadataElement platformMetadata, String name, ProductData data, String unit) {
        MetadataAttribute attribute = PrismProductDirectory.createAttribute(name, data);
        if (unit != null) {
            attribute.setUnit(unit);
        }
        platformMetadata.addAttribute(attribute);
        return attribute;
    }

    private ImageInputStream createInputStream(String fileName) throws IOException {
        return new FileImageInputStream(new File(this._baseDir, fileName));
    }
}

