/*
 * Decompiled with CFR 0.152.
 */
package org.esa.nest.gpf.geometric;

import com.bc.ceres.core.ProgressMonitor;
import java.awt.Rectangle;
import java.io.File;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.math3.util.FastMath;
import org.esa.beam.framework.datamodel.Band;
import org.esa.beam.framework.datamodel.GeoPos;
import org.esa.beam.framework.datamodel.MetadataElement;
import org.esa.beam.framework.datamodel.Product;
import org.esa.beam.framework.datamodel.ProductData;
import org.esa.beam.framework.datamodel.RasterDataNode;
import org.esa.beam.framework.datamodel.TiePointGrid;
import org.esa.beam.framework.dataop.resamp.ResamplingFactory;
import org.esa.beam.framework.gpf.Operator;
import org.esa.beam.framework.gpf.OperatorException;
import org.esa.beam.framework.gpf.OperatorSpi;
import org.esa.beam.framework.gpf.Tile;
import org.esa.beam.framework.gpf.annotations.OperatorMetadata;
import org.esa.beam.framework.gpf.annotations.Parameter;
import org.esa.beam.framework.gpf.annotations.SourceProduct;
import org.esa.beam.framework.gpf.annotations.TargetProduct;
import org.esa.beam.util.ProductUtils;
import org.esa.nest.dataio.dem.DEMFactory;
import org.esa.nest.dataio.dem.ElevationModel;
import org.esa.nest.dataio.dem.ElevationModelDescriptor;
import org.esa.nest.dataio.dem.ElevationModelRegistry;
import org.esa.nest.dataio.dem.FileElevationModel;
import org.esa.nest.gpf.geometric.RangeDopplerGeocodingOp;
import org.esa.nest.gpf.geometric.SARGeocoding;
import org.esa.nest.gpf.geometric.SARUtils;
import org.esa.snap.datamodel.AbstractMetadata;
import org.esa.snap.datamodel.OrbitStateVector;
import org.esa.snap.eo.GeoUtils;
import org.esa.snap.gpf.InputProductValidator;
import org.esa.snap.gpf.OperatorUtils;
import org.esa.snap.gpf.ReaderUtils;
import org.esa.snap.gpf.TileIndex;
import org.esa.snap.util.Maths;

@OperatorMetadata(alias="Terrain-Flattening", category="SAR Processing/Radiometric", authors="Jun Lu, Luis Veci", copyright="Copyright (C) 2014 by Array Systems Computing Inc.", description="Terrain Flattening", internal=true)
public final class TerrainFlatteningOp
extends Operator {
    @SourceProduct(alias="source")
    private Product sourceProduct;
    @TargetProduct
    private Product targetProduct;
    @Parameter(description="The list of source bands.", alias="sourceBands", itemAlias="band", rasterDataNodeType=Band.class, label="Source Bands")
    private String[] sourceBandNames;
    @Parameter(valueSet={"ACE", "GETASSE30", "SRTM 3Sec", "ASTER 1sec GDEM"}, description="The digital elevation model.", defaultValue="SRTM 3Sec", label="Digital Elevation Model")
    private String demName = "SRTM 3Sec";
    @Parameter(valueSet={"NEAREST_NEIGHBOUR", "BILINEAR_INTERPOLATION", "CUBIC_CONVOLUTION"}, defaultValue="BILINEAR_INTERPOLATION", label="DEM Resampling Method")
    private String demResamplingMethod = "BILINEAR_INTERPOLATION";
    @Parameter(label="External DEM")
    private File externalDEMFile = null;
    @Parameter(label="DEM No Data Value", defaultValue="0")
    private double externalDEMNoDataValue = 0.0;
    @Parameter(defaultValue="false", label="Output Simulated Image")
    private boolean outputSimulatedImage = false;
    private ElevationModel dem = null;
    private FileElevationModel fileElevationModel = null;
    private TiePointGrid latitudeTPG = null;
    private TiePointGrid longitudeTPG = null;
    private int sourceImageWidth = 0;
    private int sourceImageHeight = 0;
    private boolean srgrFlag = false;
    private boolean isElevationModelAvailable = false;
    private boolean overlapComputed = false;
    private double rangeSpacing = 0.0;
    private double azimuthSpacing = 0.0;
    private double firstLineUTC = 0.0;
    private double lastLineUTC = 0.0;
    private double lineTimeInterval = 0.0;
    private double nearEdgeSlantRange = 0.0;
    private double wavelength = 0.0;
    private float demNoDataValue = 0.0f;
    private SARGeocoding.Orbit orbit = null;
    private int polyDegree = 2;
    private double noDataValue = 0.0;
    private double beta0 = 0.0;
    private int tileSize = 100;
    private float tileOverlapPercentage = 0.0f;
    private OrbitStateVector[] orbitStateVectors = null;
    private AbstractMetadata.SRGRCoefficientList[] srgrConvParams = null;
    private Band[] targetBands = null;
    protected final HashMap<Band, Band> targetBandToSourceBandMap = new HashMap(2);
    private boolean nearRangeOnLeft = true;

    public void initialize() throws OperatorException {
        try {
            InputProductValidator validator = new InputProductValidator(this.sourceProduct);
            validator.checkIfMapProjected();
            validator.checkIfNotCalibrated();
            this.getMetadata();
            this.getTiePointGrid();
            this.getSourceImageDimension();
            this.computeSensorPositionsAndVelocities();
            this.createTargetProduct();
            if (this.externalDEMFile == null) {
                DEMFactory.checkIfDEMInstalled((String)this.demName);
            }
            DEMFactory.validateDEM((String)this.demName, (Product)this.sourceProduct);
            this.noDataValue = this.sourceProduct.getBands()[0].getNoDataValue();
            this.beta0 = this.azimuthSpacing * this.rangeSpacing;
        }
        catch (Throwable e) {
            OperatorUtils.catchOperatorException((String)this.getId(), (Throwable)e);
        }
    }

    public synchronized void dispose() {
        if (this.dem != null) {
            this.dem.dispose();
            this.dem = null;
        }
        if (this.fileElevationModel != null) {
            this.fileElevationModel.dispose();
        }
    }

    private void getMetadata() throws Exception {
        MetadataElement absRoot = AbstractMetadata.getAbstractedMetadata((Product)this.sourceProduct);
        this.srgrFlag = AbstractMetadata.getAttributeBoolean((MetadataElement)absRoot, (String)"srgr_flag");
        this.wavelength = SARUtils.getRadarFrequency((MetadataElement)absRoot);
        this.rangeSpacing = AbstractMetadata.getAttributeDouble((MetadataElement)absRoot, (String)"range_spacing");
        this.azimuthSpacing = AbstractMetadata.getAttributeDouble((MetadataElement)absRoot, (String)"azimuth_spacing");
        this.firstLineUTC = AbstractMetadata.parseUTC((String)absRoot.getAttributeString("first_line_time")).getMJD();
        this.lastLineUTC = AbstractMetadata.parseUTC((String)absRoot.getAttributeString("last_line_time")).getMJD();
        this.lineTimeInterval = absRoot.getAttributeDouble("line_time_interval") / 86400.0;
        this.orbitStateVectors = AbstractMetadata.getOrbitStateVectors((MetadataElement)absRoot);
        if (this.srgrFlag) {
            this.srgrConvParams = AbstractMetadata.getSRGRCoefficients((MetadataElement)absRoot);
        } else {
            this.nearEdgeSlantRange = AbstractMetadata.getAttributeDouble((MetadataElement)absRoot, (String)"slant_range_to_first_pixel");
        }
        String mission = RangeDopplerGeocodingOp.getMissionType(absRoot);
        String pass = absRoot.getAttributeString("PASS");
        if (mission.equals("RS2") && pass.contains("DESCENDING")) {
            this.nearRangeOnLeft = false;
        }
    }

    private void getSourceImageDimension() {
        this.sourceImageWidth = this.sourceProduct.getSceneRasterWidth();
        this.sourceImageHeight = this.sourceProduct.getSceneRasterHeight();
    }

    private void computeSensorPositionsAndVelocities() {
        this.orbit = new SARGeocoding.Orbit(this.orbitStateVectors, this.polyDegree, this.firstLineUTC, this.lineTimeInterval, this.sourceImageHeight);
    }

    private void getTiePointGrid() {
        this.latitudeTPG = OperatorUtils.getLatitude((Product)this.sourceProduct);
        if (this.latitudeTPG == null) {
            throw new OperatorException("Product without latitude tie point grid");
        }
        this.longitudeTPG = OperatorUtils.getLongitude((Product)this.sourceProduct);
        if (this.longitudeTPG == null) {
            throw new OperatorException("Product without longitude tie point grid");
        }
    }

    private void createTargetProduct() {
        this.targetProduct = new Product(this.sourceProduct.getName(), this.sourceProduct.getProductType(), this.sourceImageWidth, this.sourceImageHeight);
        this.addSelectedBands();
        ProductUtils.copyProductNodes((Product)this.sourceProduct, (Product)this.targetProduct);
        MetadataElement absTgt = AbstractMetadata.getAbstractedMetadata((Product)this.targetProduct);
        if (this.externalDEMFile != null && this.fileElevationModel == null) {
            AbstractMetadata.setAttribute((MetadataElement)absTgt, (String)"DEM", (String)this.externalDEMFile.getPath());
        } else {
            AbstractMetadata.setAttribute((MetadataElement)absTgt, (String)"DEM", (String)this.demName);
        }
        absTgt.setAttributeString("DEM resampling method", this.demResamplingMethod);
        absTgt.setAttributeInt("abs_calibration_flag", 1);
        if (this.externalDEMFile != null) {
            absTgt.setAttributeDouble("external DEM no data value", this.externalDEMNoDataValue);
        }
        this.targetProduct.setPreferredTileSize(this.targetProduct.getSceneRasterWidth(), this.tileSize);
    }

    private void addSelectedBands() {
        int i;
        Band[] sourceBands = OperatorUtils.getSourceBands((Product)this.sourceProduct, (String[])this.sourceBandNames);
        MetadataElement absRoot = AbstractMetadata.getAbstractedMetadata((Product)this.sourceProduct);
        for (i = 0; i < sourceBands.length; ++i) {
            String tgtUnit;
            String tgtBandName;
            Band srcBand = sourceBands[i];
            String srcBandName = srcBand.getName();
            if (!srcBandName.contains("Beta")) {
                throw new OperatorException("Please select beta0 bands only");
            }
            String unit = srcBand.getUnit();
            if (unit == null) {
                throw new OperatorException("band " + srcBandName + " requires a unit");
            }
            if (unit.contains("db")) {
                throw new OperatorException("Terrain flattening of bands in dB is not supported");
            }
            if (unit.contains("phase")) continue;
            if (unit.contains("real") || unit.contains("imaginary")) {
                tgtBandName = srcBandName;
                tgtUnit = unit;
            } else {
                String pol = OperatorUtils.getBandPolarization((String)srcBandName, (MetadataElement)absRoot);
                tgtBandName = "Gamma0";
                if (pol != null && !pol.isEmpty()) {
                    tgtBandName = "Gamma0_" + pol.toUpperCase();
                }
                tgtUnit = "intensity";
            }
            if (this.targetProduct.getBand(tgtBandName) != null) continue;
            Band tgtBand = this.targetProduct.addBand(tgtBandName, 30);
            tgtBand.setUnit(tgtUnit);
            this.targetBandToSourceBandMap.put(tgtBand, srcBand);
        }
        if (this.outputSimulatedImage) {
            Band tgtBand = this.targetProduct.addBand("simulatedImage", 30);
            tgtBand.setUnit("Ratio");
        }
        this.targetBands = this.targetProduct.getBands();
        for (i = 0; i < this.targetBands.length; ++i) {
            if (!this.targetBands[i].getUnit().equals("real")) continue;
            String trgBandName = this.targetBands[i].getName();
            int idx = trgBandName.indexOf("_");
            String suffix = "";
            if (idx != -1) {
                suffix = trgBandName.substring(trgBandName.indexOf("_"));
            }
            ReaderUtils.createVirtualIntensityBand((Product)this.targetProduct, (Band)this.targetBands[i], (Band)this.targetBands[i + 1], (String)"Gamma0", (String)suffix);
        }
    }

    public void computeTileStack(Map<Band, Tile> targetTiles, Rectangle targetRectangle, ProgressMonitor pm) throws OperatorException {
        try {
            double[][] simulatedImage;
            int h;
            int w;
            int y0;
            int x0;
            boolean validSimulation;
            if (!this.isElevationModelAvailable) {
                this.getElevationModel();
            }
            if (!this.overlapComputed) {
                this.computeTileOverlapPercentage(this.tileSize);
            }
            if (!(validSimulation = this.generateSimulatedImage(x0 = targetRectangle.x, y0 = targetRectangle.y, w = targetRectangle.width, h = targetRectangle.height, simulatedImage = new double[h][w]))) {
                return;
            }
            this.outputNormalizedImage(x0, y0, w, h, simulatedImage, targetTiles, targetRectangle);
        }
        catch (Throwable e) {
            OperatorUtils.catchOperatorException((String)this.getId(), (Throwable)e);
        }
    }

    private boolean generateSimulatedImage(int x0, int y0, int w, int h, double[][] simulatedImage) {
        try {
            int ymin = 0;
            int ymax = 0;
            if (this.tileOverlapPercentage >= 0.0f) {
                ymin = Math.max(y0 - (int)((float)this.tileSize * this.tileOverlapPercentage), 0);
                ymax = y0 + h;
            } else {
                ymin = y0;
                ymax = y0 + h + (int)((float)this.tileSize * Math.abs(this.tileOverlapPercentage));
            }
            TerrainData terrainData = new TerrainData(w, ymax - ymin);
            boolean valid = this.getLocalDEM(x0, ymin, w, ymax - ymin, terrainData);
            if (!valid) {
                return false;
            }
            double[] earthPoint = new double[3];
            double[] sensorPos = new double[3];
            for (int y = ymin; y < ymax; ++y) {
                int i;
                int x;
                double maxElevAngle;
                double[] azimuthIndex = new double[w];
                double[] rangeIndex = new double[w];
                double[] illuminatedArea = new double[w];
                double[] elevationAngle = new double[w];
                boolean[] savePixel = new boolean[w];
                for (int x2 = x0; x2 < x0 + w; ++x2) {
                    int i2 = x2 - x0;
                    int yy = y - ymin + 1;
                    int xx = x2 - x0 + 1;
                    double alt = terrainData.localDEM[yy][xx];
                    if (alt == (double)this.demNoDataValue) {
                        savePixel[i2] = false;
                        continue;
                    }
                    GeoUtils.geo2xyzWGS84((double)terrainData.latPixels[yy][xx], (double)terrainData.lonPixels[yy][xx], (double)alt, (double[])earthPoint);
                    double zeroDopplerTime = SARGeocoding.getEarthPointZeroDopplerTime((double)this.firstLineUTC, (double)this.lineTimeInterval, (double)this.wavelength, (double[])earthPoint, (double[][])this.orbit.sensorPosition, (double[][])this.orbit.sensorVelocity);
                    double slantRange = SARGeocoding.computeSlantRange((double)zeroDopplerTime, (SARGeocoding.Orbit)this.orbit, (double[])earthPoint, (double[])sensorPos);
                    double zeroDopplerTimeWithoutBias = zeroDopplerTime + slantRange / 2.59020683712E13;
                    azimuthIndex[i2] = (zeroDopplerTimeWithoutBias - this.firstLineUTC) / this.lineTimeInterval;
                    slantRange = SARGeocoding.computeSlantRange((double)zeroDopplerTimeWithoutBias, (SARGeocoding.Orbit)this.orbit, (double[])earthPoint, (double[])sensorPos);
                    rangeIndex[i2] = SARGeocoding.computeRangeIndex((boolean)this.srgrFlag, (int)this.sourceImageWidth, (double)this.firstLineUTC, (double)this.lastLineUTC, (double)this.rangeSpacing, (double)zeroDopplerTimeWithoutBias, (double)slantRange, (double)this.nearEdgeSlantRange, (AbstractMetadata.SRGRCoefficientList[])this.srgrConvParams);
                    if (rangeIndex[i2] <= 0.0) continue;
                    if (!this.nearRangeOnLeft) {
                        rangeIndex[i2] = (double)(this.sourceImageWidth - 1) - rangeIndex[i2];
                    }
                    LocalGeometry localGeometry = new LocalGeometry(x2, y, earthPoint, sensorPos, terrainData, xx, yy);
                    illuminatedArea[i2] = this.computeLocalIlluminatedArea(x0, ymin, x2, y, localGeometry, terrainData.localDEM, this.demNoDataValue);
                    if (illuminatedArea[i2] == this.noDataValue) {
                        savePixel[i2] = false;
                        continue;
                    }
                    elevationAngle[i2] = TerrainFlatteningOp.computeElevationAngle(slantRange, earthPoint, sensorPos);
                    savePixel[i2] = rangeIndex[i2] >= (double)x0 && rangeIndex[i2] < (double)(x0 + w) && azimuthIndex[i2] > (double)(y0 - 1) && azimuthIndex[i2] < (double)(y0 + h);
                }
                if (this.nearRangeOnLeft) {
                    maxElevAngle = 0.0;
                    for (x = x0; x < x0 + w; ++x) {
                        i = x - x0;
                        if (!savePixel[i] || !(elevationAngle[i] > maxElevAngle)) continue;
                        maxElevAngle = elevationAngle[i];
                        this.saveLocalIlluminatedArea(x0, y0, w, h, illuminatedArea[i], azimuthIndex[i], rangeIndex[i], simulatedImage);
                    }
                    continue;
                }
                maxElevAngle = 0.0;
                for (x = x0 + w - 1; x >= x0; --x) {
                    i = x - x0;
                    if (!savePixel[i] || !(elevationAngle[i] > maxElevAngle)) continue;
                    maxElevAngle = elevationAngle[i];
                    this.saveLocalIlluminatedArea(x0, y0, w, h, illuminatedArea[i], azimuthIndex[i], rangeIndex[i], simulatedImage);
                }
            }
        }
        catch (Throwable e) {
            OperatorUtils.catchOperatorException((String)this.getId(), (Throwable)e);
        }
        return true;
    }

    private void outputNormalizedImage(int x0, int y0, int w, int h, double[][] simulatedImage, Map<Band, Tile> targetTiles, Rectangle targetRectangle) {
        for (Band tgtBand : this.targetBands) {
            Tile targetTile = targetTiles.get(tgtBand);
            ProductData targetData = targetTile.getDataBuffer();
            TileIndex trgIndex = new TileIndex(targetTile);
            String unit = tgtBand.getUnit();
            Band srcBand = null;
            Tile sourceTile = null;
            ProductData sourceData = null;
            if (!unit.contains("Ratio")) {
                srcBand = this.targetBandToSourceBandMap.get(tgtBand);
                sourceTile = this.getSourceTile((RasterDataNode)srcBand, targetRectangle);
                sourceData = sourceTile.getDataBuffer();
            }
            for (int y = y0; y < y0 + h; ++y) {
                int yy = y - y0;
                trgIndex.calculateStride(y);
                for (int x = x0; x < x0 + w; ++x) {
                    int xx = x - x0;
                    int idx = trgIndex.getIndex(x);
                    if (simulatedImage[yy][xx] != this.noDataValue && simulatedImage[yy][xx] != 0.0) {
                        double v;
                        if (unit.contains("amplitude")) {
                            v = sourceData.getElemDoubleAt(idx);
                            targetData.setElemDoubleAt(idx, v * v / simulatedImage[yy][xx]);
                            continue;
                        }
                        if (unit.contains("intensity")) {
                            v = sourceData.getElemDoubleAt(idx);
                            targetData.setElemDoubleAt(idx, v / simulatedImage[yy][xx]);
                            continue;
                        }
                        if (unit.contains("real") || unit.contains("imaginary")) {
                            v = sourceData.getElemDoubleAt(idx);
                            targetData.setElemDoubleAt(idx, v / Math.sqrt(simulatedImage[yy][xx]));
                            continue;
                        }
                        if (!unit.contains("Ratio")) continue;
                        targetData.setElemDoubleAt(idx, simulatedImage[yy][xx]);
                        continue;
                    }
                    targetData.setElemDoubleAt(idx, this.noDataValue);
                }
            }
        }
    }

    private synchronized void getElevationModel() throws Exception {
        if (this.isElevationModelAvailable) {
            return;
        }
        if (this.externalDEMFile != null && this.fileElevationModel == null) {
            this.fileElevationModel = new FileElevationModel(this.externalDEMFile, ResamplingFactory.createResampling((String)this.demResamplingMethod), Float.valueOf((float)this.externalDEMNoDataValue));
            this.demNoDataValue = (float)this.externalDEMNoDataValue;
            this.demName = this.externalDEMFile.getPath();
        } else {
            ElevationModelRegistry elevationModelRegistry = ElevationModelRegistry.getInstance();
            ElevationModelDescriptor demDescriptor = elevationModelRegistry.getDescriptor(this.demName);
            if (demDescriptor == null) {
                throw new OperatorException("The DEM '" + this.demName + "' is not supported.");
            }
            if (demDescriptor.isInstallingDem()) {
                throw new OperatorException("The DEM '" + this.demName + "' is currently being installed.");
            }
            this.dem = demDescriptor.createDem(ResamplingFactory.createResampling((String)this.demResamplingMethod));
            if (this.dem == null) {
                throw new OperatorException("The DEM '" + this.demName + "' has not been installed.");
            }
            this.demNoDataValue = this.dem.getDescriptor().getNoDataValue();
        }
        this.isElevationModelAvailable = true;
    }

    private synchronized void computeTileOverlapPercentage(int tileSize) throws Exception {
        int y;
        if (this.overlapComputed) {
            return;
        }
        int x = this.sourceImageWidth / 2;
        double[] earthPoint = new double[3];
        double[] sensorPos = new double[3];
        GeoPos geoPos = new GeoPos();
        double alt = 0.0;
        for (y = tileSize - 1; y < this.sourceImageHeight; ++y) {
            geoPos.setLocation((float)this.latitudeTPG.getPixelDouble(x, y), (float)this.longitudeTPG.getPixelDouble(x, y));
            alt = this.externalDEMFile == null ? this.dem.getElevation(geoPos) : this.fileElevationModel.getElevation(geoPos);
            if (alt != (double)this.demNoDataValue) break;
        }
        GeoUtils.geo2xyzWGS84((double)this.latitudeTPG.getPixelDouble(x, y), (double)this.longitudeTPG.getPixelDouble(x, y), (double)alt, (double[])earthPoint);
        double zeroDopplerTime = SARGeocoding.getEarthPointZeroDopplerTime((double)this.firstLineUTC, (double)this.lineTimeInterval, (double)this.wavelength, (double[])earthPoint, (double[][])this.orbit.sensorPosition, (double[][])this.orbit.sensorVelocity);
        double slantRange = SARGeocoding.computeSlantRange((double)zeroDopplerTime, (SARGeocoding.Orbit)this.orbit, (double[])earthPoint, (double[])sensorPos);
        double zeroDopplerTimeWithoutBias = zeroDopplerTime + slantRange / 2.59020683712E13;
        int azimuthIndex = (int)((zeroDopplerTimeWithoutBias - this.firstLineUTC) / this.lineTimeInterval + 0.5);
        this.tileOverlapPercentage = (float)(azimuthIndex - y) / (float)tileSize;
        this.tileOverlapPercentage = (double)this.tileOverlapPercentage >= 0.0 ? (float)((double)this.tileOverlapPercentage + 0.05) : (float)((double)this.tileOverlapPercentage - 0.05);
        this.overlapComputed = true;
    }

    private boolean getLocalDEM(int x0, int y0, int tileWidth, int tileHeight, TerrainData terrainData) throws Exception {
        GeoPos geoPos = new GeoPos();
        int maxY = y0 + tileHeight + 1;
        int maxX = x0 + tileWidth + 1;
        boolean valid = false;
        for (int y = y0 - 1; y < maxY; ++y) {
            int yy = y - y0 + 1;
            for (int x = x0 - 1; x < maxX; ++x) {
                int xx = x - x0 + 1;
                double lat = this.latitudeTPG.getPixelFloat((float)x + 0.5f, (float)y + 0.5f);
                double lon = this.longitudeTPG.getPixelFloat((float)x + 0.5f, (float)y + 0.5f);
                geoPos.setLocation((float)lat, (float)lon);
                double alt = this.externalDEMFile == null ? this.dem.getElevation(geoPos) : this.fileElevationModel.getElevation(geoPos);
                terrainData.localDEM[yy][xx] = alt;
                terrainData.latPixels[yy][xx] = lat;
                terrainData.lonPixels[yy][xx] = lon;
                if (alt == (double)this.demNoDataValue) continue;
                valid = true;
            }
        }
        if (this.fileElevationModel != null) {
            // empty if block
        }
        return valid;
    }

    private void saveLocalIlluminatedArea(int x0, int y0, int w, int h, double illuminatedArea, double azimuthIndex, double rangeIndex, double[][] simulatedImage) {
        int ia0 = (int)azimuthIndex;
        int ia1 = ia0 + 1;
        int ir0 = (int)rangeIndex;
        int ir1 = ir0 + 1;
        double wr = rangeIndex - (double)ir0;
        double wa = azimuthIndex - (double)ia0;
        double wac = 1.0 - wa;
        if (ir0 >= x0) {
            double wrc = 1.0 - wr;
            if (ia0 >= y0) {
                double[] dArray = simulatedImage[ia0 - y0];
                int n = ir0 - x0;
                dArray[n] = dArray[n] + wrc * wac * illuminatedArea / this.beta0;
            }
            if (ia1 < y0 + h) {
                double[] dArray = simulatedImage[ia1 - y0];
                int n = ir0 - x0;
                dArray[n] = dArray[n] + wrc * wa * illuminatedArea / this.beta0;
            }
        }
        if (ir1 < x0 + w) {
            if (ia0 >= y0) {
                double[] dArray = simulatedImage[ia0 - y0];
                int n = ir1 - x0;
                dArray[n] = dArray[n] + wr * wac * illuminatedArea / this.beta0;
            }
            if (ia1 < y0 + h) {
                double[] dArray = simulatedImage[ia1 - y0];
                int n = ir1 - x0;
                dArray[n] = dArray[n] + wr * wa * illuminatedArea / this.beta0;
            }
        }
    }

    private static double computeElevationAngle(double slantRange, double[] earthPoint, double[] sensorPos) {
        double H2 = sensorPos[0] * sensorPos[0] + sensorPos[1] * sensorPos[1] + sensorPos[2] * sensorPos[2];
        double R2 = earthPoint[0] * earthPoint[0] + earthPoint[1] * earthPoint[1] + earthPoint[2] * earthPoint[2];
        return FastMath.acos((double)((slantRange * slantRange + H2 - R2) / (2.0 * slantRange * Math.sqrt(H2)))) * 57.29577951308232;
    }

    private double computeLocalIlluminatedArea(int xMin, int yMin, int x, int y, LocalGeometry lg, double[][] localDEM, double demNoDataValue) {
        int yy = y - yMin + 1;
        int xx = x - xMin + 1;
        double h00 = localDEM[yy][xx];
        double h01 = localDEM[yy - 1][xx];
        double h10 = localDEM[yy][xx + 1];
        double h11 = localDEM[yy - 1][xx + 1];
        if (h00 == demNoDataValue || h01 == demNoDataValue || h10 == demNoDataValue || h11 == demNoDataValue) {
            return this.noDataValue;
        }
        double[] t00 = new double[3];
        double[] t01 = new double[3];
        double[] t10 = new double[3];
        double[] t11 = new double[3];
        GeoUtils.geo2xyzWGS84((double)lg.t00Lat, (double)lg.t00Lon, (double)h00, (double[])t00);
        GeoUtils.geo2xyzWGS84((double)lg.t01Lat, (double)lg.t01Lon, (double)h01, (double[])t01);
        GeoUtils.geo2xyzWGS84((double)lg.t10Lat, (double)lg.t10Lon, (double)h10, (double[])t10);
        GeoUtils.geo2xyzWGS84((double)lg.t11Lat, (double)lg.t11Lon, (double)h11, (double[])t11);
        double[] s = new double[]{lg.sensorPos[0] - lg.centerPoint[0], lg.sensorPos[1] - lg.centerPoint[1], lg.sensorPos[2] - lg.centerPoint[2]};
        Maths.normalizeVector((double[])s);
        double t00s = Maths.innerProduct((double[])t00, (double[])s);
        double t01s = Maths.innerProduct((double[])t01, (double[])s);
        double t10s = Maths.innerProduct((double[])t10, (double[])s);
        double t11s = Maths.innerProduct((double[])t11, (double[])s);
        double[] p00 = new double[]{t00[0] - t00s * s[0], t00[1] - t00s * s[1], t00[2] - t00s * s[2]};
        double[] p01 = new double[]{t01[0] - t01s * s[0], t01[1] - t01s * s[1], t01[2] - t01s * s[2]};
        double[] p10 = new double[]{t10[0] - t10s * s[0], t10[1] - t10s * s[1], t10[2] - t10s * s[2]};
        double[] p11 = new double[]{t11[0] - t11s * s[0], t11[1] - t11s * s[1], t11[2] - t11s * s[2]};
        double p00p01 = TerrainFlatteningOp.distance(p00, p01);
        double p00p10 = TerrainFlatteningOp.distance(p00, p10);
        double p11p01 = TerrainFlatteningOp.distance(p11, p01);
        double p11p10 = TerrainFlatteningOp.distance(p11, p10);
        double p10p01 = TerrainFlatteningOp.distance(p10, p01);
        double h1 = 0.5 * (p00p01 + p00p10 + p10p01);
        double h2 = 0.5 * (p11p01 + p11p10 + p10p01);
        return Math.sqrt(h1 * (h1 - p00p01) * (h1 - p00p10) * (h1 - p10p01)) + Math.sqrt(h2 * (h2 - p11p01) * (h2 - p11p10) * (h2 - p10p01));
    }

    private static double distance(double[] p1, double[] p2) {
        return Math.sqrt((p1[0] - p2[0]) * (p1[0] - p2[0]) + (p1[1] - p2[1]) * (p1[1] - p2[1]) + (p1[2] - p2[2]) * (p1[2] - p2[2]));
    }

    public static class Spi
    extends OperatorSpi {
        public Spi() {
            super(TerrainFlatteningOp.class);
        }
    }

    private static class TerrainData {
        final double[][] localDEM;
        final double[][] latPixels;
        final double[][] lonPixels;

        public TerrainData(int w, int h) {
            this.localDEM = new double[h + 2][w + 2];
            this.latPixels = new double[h + 2][w + 2];
            this.lonPixels = new double[h + 2][w + 2];
        }
    }

    public static class LocalGeometry {
        public final double t00Lat;
        public final double t00Lon;
        public final double t01Lat;
        public final double t01Lon;
        public final double t10Lat;
        public final double t10Lon;
        public final double t11Lat;
        public final double t11Lon;
        public final double[] sensorPos;
        public final double[] centerPoint;

        public LocalGeometry(int x, int y, double[] earthPoint, double[] sensPos, TerrainData terrainData, int xx, int yy) {
            this.t00Lat = terrainData.latPixels[yy][xx];
            this.t00Lon = terrainData.lonPixels[yy][xx];
            this.t01Lat = terrainData.latPixels[yy - 1][xx];
            this.t01Lon = terrainData.lonPixels[yy - 1][xx];
            this.t10Lat = terrainData.latPixels[yy][xx + 1];
            this.t10Lon = terrainData.lonPixels[yy][xx + 1];
            this.t11Lat = terrainData.latPixels[yy + 1][xx + 1];
            this.t11Lon = terrainData.lonPixels[yy + 1][xx + 1];
            this.centerPoint = earthPoint;
            this.sensorPos = sensPos;
        }
    }
}

