/*
 * Decompiled with CFR 0.152.
 */
package org.esa.beam.framework.datamodel;

import java.awt.Dimension;
import java.awt.Rectangle;
import org.esa.beam.framework.dataio.ProductSubsetDef;
import org.esa.beam.framework.datamodel.AbstractGeoCoding;
import org.esa.beam.framework.datamodel.GeoPos;
import org.esa.beam.framework.datamodel.PixelPos;
import org.esa.beam.framework.datamodel.Product;
import org.esa.beam.framework.datamodel.Scene;
import org.esa.beam.framework.datamodel.TiePointGrid;
import org.esa.beam.framework.dataop.maptransf.Datum;
import org.esa.beam.util.Debug;
import org.esa.beam.util.Guardian;
import org.esa.beam.util.math.FXYSum;
import org.esa.beam.util.math.MathUtils;

public class TiePointGeoCoding
extends AbstractGeoCoding {
    private static final double ABS_ERROR_LIMIT = 0.5;
    private static final int MAX_NUM_POINTS_PER_TILE = 1000;
    private final TiePointGrid latGrid;
    private final TiePointGrid lonGrid;
    private final Approximation[] approximations;
    private final Datum datum;
    private boolean normalized;
    private float normalizedLonMin;
    private float normalizedLonMax;
    private float latMin;
    private float latMax;
    private float overlapStart;
    private float overlapEnd;

    public TiePointGeoCoding(TiePointGrid latGrid, TiePointGrid lonGrid) {
        this(latGrid, lonGrid, Datum.WGS_84);
    }

    public TiePointGeoCoding(TiePointGrid latGrid, TiePointGrid lonGrid, Datum datum) {
        Guardian.assertNotNull("latGrid", latGrid);
        Guardian.assertNotNull("lonGrid", lonGrid);
        Guardian.assertNotNull("datum", datum);
        if (latGrid.getRasterWidth() != lonGrid.getRasterWidth() || latGrid.getRasterHeight() != lonGrid.getRasterHeight() || latGrid.getOffsetX() != lonGrid.getOffsetX() || latGrid.getOffsetY() != lonGrid.getOffsetY() || latGrid.getSubSamplingX() != lonGrid.getSubSamplingX() || latGrid.getSubSamplingY() != lonGrid.getSubSamplingY()) {
            throw new IllegalArgumentException("latGrid is not compatible with lonGrid");
        }
        this.latGrid = latGrid;
        this.lonGrid = lonGrid;
        this.datum = datum;
        TiePointGrid normalizedLonGrid = this.initNormalizedLonGrid();
        this.initLatLonMinMax(normalizedLonGrid);
        this.approximations = this.initApproximations(normalizedLonGrid);
    }

    @Override
    public Datum getDatum() {
        return this.datum;
    }

    @Override
    public boolean isCrossingMeridianAt180() {
        return this.normalized;
    }

    public int getNumApproximations() {
        return this.approximations != null ? this.approximations.length : 0;
    }

    public Approximation getApproximation(int index) {
        return this.approximations[index];
    }

    @Override
    public boolean canGetGeoPos() {
        return true;
    }

    @Override
    public boolean canGetPixelPos() {
        return this.approximations != null;
    }

    public TiePointGrid getLatGrid() {
        return this.latGrid;
    }

    public TiePointGrid getLonGrid() {
        return this.lonGrid;
    }

    @Override
    public GeoPos getGeoPos(PixelPos pixelPos, GeoPos geoPos) {
        if (geoPos == null) {
            geoPos = new GeoPos();
        }
        geoPos.lat = this.latGrid.getPixelFloat(pixelPos.x, pixelPos.y);
        geoPos.lon = this.lonGrid.getPixelFloat(pixelPos.x, pixelPos.y);
        return geoPos;
    }

    @Override
    public PixelPos getPixelPos(GeoPos geoPos, PixelPos pixelPos) {
        if (this.approximations != null) {
            float lat = TiePointGeoCoding.normalizeLat(geoPos.lat);
            float lon = this.normalizeLon(geoPos.lon);
            if (pixelPos == null) {
                pixelPos = new PixelPos();
            }
            if (this.isValidGeoPos(lat, lon)) {
                float squareDistance;
                float tempLon;
                Approximation renormalizedApproximation;
                Approximation approximation = TiePointGeoCoding.getBestApproximation(this.approximations, lat, lon);
                if (lon >= this.overlapStart && lon <= this.overlapEnd && (renormalizedApproximation = this.findRenormalizedApproximation(lat, tempLon = lon + 360.0f, squareDistance = approximation != null ? approximation.getSquareDistance(lat, lon) : Float.MAX_VALUE)) != null) {
                    approximation = renormalizedApproximation;
                    lon = tempLon;
                }
                if (approximation != null) {
                    lat = (float)TiePointGeoCoding.rescaleLatitude(lat);
                    lon = (float)TiePointGeoCoding.rescaleLongitude(lon, approximation.getCenterLon());
                    pixelPos.x = (float)approximation.getFX().computeZ(lat, lon);
                    pixelPos.y = (float)approximation.getFY().computeZ(lat, lon);
                } else {
                    pixelPos.setInvalid();
                }
            } else {
                pixelPos.setInvalid();
            }
        }
        return pixelPos;
    }

    private boolean isValidGeoPos(float lat, float lon) {
        return !Float.isNaN(lat) && !Float.isNaN(lon);
    }

    public static float normalizeLat(float lat) {
        if (lat < -90.0f || lat > 90.0f) {
            return Float.NaN;
        }
        return lat;
    }

    public final float normalizeLon(float lon) {
        if (lon < -180.0f || lon > 180.0f) {
            return Float.NaN;
        }
        float normalizedLon = lon;
        if (normalizedLon < this.normalizedLonMin) {
            normalizedLon += 360.0f;
        }
        if (normalizedLon < this.normalizedLonMin || normalizedLon > this.normalizedLonMax) {
            return Float.NaN;
        }
        return normalizedLon;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TiePointGeoCoding that = (TiePointGeoCoding)o;
        if (!this.latGrid.equals(that.latGrid)) {
            return false;
        }
        return this.lonGrid.equals(that.lonGrid);
    }

    public int hashCode() {
        int result = this.latGrid.hashCode();
        result = 31 * result + this.lonGrid.hashCode();
        return result;
    }

    @Override
    public void dispose() {
    }

    private TiePointGrid initNormalizedLonGrid() {
        int w = this.lonGrid.getRasterWidth();
        int h = this.lonGrid.getRasterHeight();
        boolean westNormalized = false;
        boolean eastNormalized = false;
        float[] longitudes = this.lonGrid.getTiePoints();
        int numValues = longitudes.length;
        float[] normalizedLongitudes = new float[numValues];
        System.arraycopy(longitudes, 0, normalizedLongitudes, 0, numValues);
        float lonDeltaMax = 0.0f;
        for (int y = 0; y < h; ++y) {
            for (int x = 0; x < w; ++x) {
                int index = x + y * w;
                float p2 = normalizedLongitudes[index];
                float p1 = x == 0 && y == 0 ? normalizedLongitudes[index] : (x == 0 ? normalizedLongitudes[x + (y - 1) * w] : normalizedLongitudes[index - 1]);
                float lonDelta = p2 - p1;
                if (lonDelta > 180.0f) {
                    westNormalized = true;
                    normalizedLongitudes[index] = p2 -= 360.0f;
                    continue;
                }
                if (lonDelta < -180.0f) {
                    eastNormalized = true;
                    normalizedLongitudes[index] = p2 += 360.0f;
                    continue;
                }
                lonDeltaMax = Math.max(lonDeltaMax, Math.abs(lonDelta));
            }
        }
        if (westNormalized) {
            int i = 0;
            while (i < numValues) {
                int n = i++;
                normalizedLongitudes[n] = normalizedLongitudes[n] + 360.0f;
            }
        }
        this.normalized = westNormalized || eastNormalized;
        TiePointGrid normalizedLonGrid = this.normalized ? new TiePointGrid(this.lonGrid.getName(), this.lonGrid.getRasterWidth(), this.lonGrid.getRasterHeight(), this.lonGrid.getOffsetX(), this.lonGrid.getOffsetY(), this.lonGrid.getSubSamplingX(), this.lonGrid.getSubSamplingY(), normalizedLongitudes, this.lonGrid.getDiscontinuity()) : this.lonGrid;
        Debug.trace("TiePointGeoCoding.westNormalized = " + westNormalized);
        Debug.trace("TiePointGeoCoding.eastNormalized = " + eastNormalized);
        Debug.trace("TiePointGeoCoding.normalized = " + this.normalized);
        Debug.trace("TiePointGeoCoding.lonDeltaMax = " + lonDeltaMax);
        return normalizedLonGrid;
    }

    private void initLatLonMinMax(TiePointGrid normalizedLonGrid) {
        float[] latPoints = this.getLatGrid().getTiePoints();
        float[] lonPoints = normalizedLonGrid.getTiePoints();
        this.normalizedLonMin = Float.MAX_VALUE;
        this.normalizedLonMax = -3.4028235E38f;
        this.latMin = Float.MAX_VALUE;
        this.latMax = -3.4028235E38f;
        for (float lonPoint : lonPoints) {
            this.normalizedLonMin = Math.min(this.normalizedLonMin, lonPoint);
            this.normalizedLonMax = Math.max(this.normalizedLonMax, lonPoint);
        }
        for (float latPoint : latPoints) {
            this.latMin = Math.min(this.latMin, latPoint);
            this.latMax = Math.max(this.latMax, latPoint);
        }
        this.overlapStart = this.normalizedLonMin;
        if (this.overlapStart < -180.0f) {
            this.overlapStart += 360.0f;
        }
        this.overlapEnd = this.normalizedLonMax;
        if (this.overlapEnd > 180.0f) {
            this.overlapEnd -= 360.0f;
        }
        Debug.trace("TiePointGeoCoding.normalizedLonMin = " + this.normalizedLonMin);
        Debug.trace("TiePointGeoCoding.normalizedLonMax = " + this.normalizedLonMax);
        Debug.trace("TiePointGeoCoding.latMin = " + this.latMin);
        Debug.trace("TiePointGeoCoding.latMax = " + this.latMax);
        Debug.trace("TiePointGeoCoding.overlapRange = " + this.overlapStart + " - " + this.overlapEnd);
    }

    private Approximation[] initApproximations(TiePointGrid normalizedLonGrid) {
        int numTiles;
        int numPoints = this.latGrid.getRasterData().getNumElems();
        int w = this.latGrid.getRasterWidth();
        int h = this.latGrid.getRasterHeight();
        if (h > 2) {
            float lonSpan = this.normalizedLonMax - this.normalizedLonMin;
            float latSpan = this.latMax - this.latMin;
            float angleSpan = Math.max(lonSpan, latSpan);
            numTiles = Math.round(angleSpan / 10.0f);
            if (numTiles < 1) {
                numTiles = 1;
            }
        } else {
            numTiles = 30;
        }
        while (numTiles > 1 && numPoints / numTiles < 10) {
            --numTiles;
        }
        Dimension tileDim = MathUtils.fitDimension(numTiles, w, h);
        int numTilesI = tileDim.width;
        int numTilesJ = tileDim.height;
        numTiles = numTilesI * numTilesJ;
        Debug.trace("TiePointGeoCoding.numTiles =  " + numTiles);
        Debug.trace("TiePointGeoCoding.numTilesI = " + numTilesI);
        Debug.trace("TiePointGeoCoding.numTilesJ = " + numTilesJ);
        Approximation[] approximations = new Approximation[numTiles];
        Rectangle[] rectangles = MathUtils.subdivideRectangle(w, h, numTilesI, numTilesJ, 1);
        for (int i = 0; i < rectangles.length; ++i) {
            Approximation approximation = this.createApproximation(normalizedLonGrid, rectangles[i]);
            if (approximation == null) {
                return null;
            }
            approximations[i] = approximation;
        }
        return approximations;
    }

    private static FXYSum getBestPolynomial(double[][] data, int[] indices) {
        FXYSum[] potentialPolynomials = new FXYSum[]{new FXYSum.Linear(), new FXYSum.BiLinear(), new FXYSum.Quadric(), new FXYSum.BiQuadric(), new FXYSum.Cubic(), new FXYSum.BiCubic(), new FXYSum(FXYSum.FXY_4TH, 4), new FXYSum(FXYSum.FXY_BI_4TH, 8)};
        double rmseMin = Double.MAX_VALUE;
        int index = -1;
        for (int i = 0; i < potentialPolynomials.length; ++i) {
            FXYSum potentialPolynomial = potentialPolynomials[i];
            int order = potentialPolynomial.getOrder();
            int numPointsRequired = order >= 0 ? (order + 2) * (order + 1) / 2 : 2 * potentialPolynomial.getNumTerms();
            if (data.length < numPointsRequired) continue;
            try {
                potentialPolynomial.approximate(data, indices);
                double rmse = potentialPolynomial.getRootMeanSquareError();
                double maxError = potentialPolynomial.getMaxError();
                if (rmse < rmseMin) {
                    index = i;
                    rmseMin = rmse;
                }
                if (!(maxError < 0.5)) continue;
                index = i;
                break;
            }
            catch (ArithmeticException e) {
                Debug.trace("Polynomial cannot be constructed due to a numerically singular or degenerate matrix:");
                Debug.trace(e);
            }
        }
        return index >= 0 ? potentialPolynomials[index] : null;
    }

    private double[][] createWarpPoints(TiePointGrid lonGrid, Rectangle subsetRect) {
        TiePointGrid latGrid = this.getLatGrid();
        int w = latGrid.getRasterWidth();
        int sw = subsetRect.width;
        int sh = subsetRect.height;
        int i1 = subsetRect.x;
        int i2 = i1 + sw - 1;
        int j1 = subsetRect.y;
        int j2 = j1 + sh - 1;
        Debug.trace("Selecting warp points for X/Y approximations");
        Debug.trace("  subset rectangle (in tie point coordinates): " + subsetRect);
        Debug.trace("  index i: " + i1 + " to " + i2);
        Debug.trace("  index j: " + j1 + " to " + j2);
        int numU = sw;
        int numV = sh;
        int stepI = 1;
        int stepJ = 1;
        boolean adjustStepI = true;
        while (numU * numV > 1000) {
            if (adjustStepI) {
                numU = sw / ++stepI;
            } else {
                numV = sh / ++stepJ;
            }
            adjustStepI = !adjustStepI;
        }
        numU = Math.max(1, numU);
        numV = Math.max(1, numV);
        if (sw % stepI != 0) {
            ++numU;
        }
        if (sh % stepJ != 0) {
            ++numV;
        }
        int m = numU * numV;
        double[][] data = new double[m][4];
        int k = 0;
        for (int v = 0; v < numV; ++v) {
            int j = j1 + v * stepJ;
            if (j > j2) {
                j = j2;
            }
            for (int u = 0; u < numU; ++u) {
                int i = i1 + u * stepI;
                if (i > i2) {
                    i = i2;
                }
                float lat = latGrid.getRasterData().getElemFloatAt(j * w + i);
                float lon = lonGrid.getRasterData().getElemFloatAt(j * w + i);
                float x = latGrid.getOffsetX() + (float)i * latGrid.getSubSamplingX();
                float y = latGrid.getOffsetY() + (float)j * latGrid.getSubSamplingY();
                data[k][0] = lat;
                data[k][1] = lon;
                data[k][2] = x;
                data[k][3] = y;
                ++k;
            }
        }
        Debug.assertTrue(k == m);
        Debug.trace("TiePointGeoCoding: numU=" + numU + ", stepI=" + stepI);
        Debug.trace("TiePointGeoCoding: numV=" + numV + ", stepJ=" + stepJ);
        return data;
    }

    private Approximation createApproximation(TiePointGrid normalizedLonGrid, Rectangle subsetRect) {
        double[][] data = this.createWarpPoints(normalizedLonGrid, subsetRect);
        float sumLat = 0.0f;
        float sumLon = 0.0f;
        for (double[] point : data) {
            sumLat = (float)((double)sumLat + point[0]);
            sumLon = (float)((double)sumLon + point[1]);
        }
        float centerLon = sumLon / (float)data.length;
        float centerLat = sumLat / (float)data.length;
        float maxSquareDistance = TiePointGeoCoding.getMaxSquareDistance(data, centerLat, centerLon);
        for (int i = 0; i < data.length; ++i) {
            data[i][0] = TiePointGeoCoding.rescaleLatitude(data[i][0]);
            data[i][1] = TiePointGeoCoding.rescaleLongitude(data[i][1], centerLon);
        }
        int[] xIndices = new int[]{0, 1, 2};
        int[] yIndices = new int[]{0, 1, 3};
        FXYSum fX = TiePointGeoCoding.getBestPolynomial(data, xIndices);
        FXYSum fY = TiePointGeoCoding.getBestPolynomial(data, yIndices);
        if (fX == null || fY == null) {
            return null;
        }
        double rmseX = fX.getRootMeanSquareError();
        double rmseY = fY.getRootMeanSquareError();
        double maxErrorX = fX.getMaxError();
        double maxErrorY = fY.getMaxError();
        Debug.trace("TiePointGeoCoding: RMSE X      = " + rmseX + ", " + (rmseX < 0.5 ? "OK" : "too large"));
        Debug.trace("TiePointGeoCoding: RMSE Y      = " + rmseY + ", " + (rmseY < 0.5 ? "OK" : "too large"));
        Debug.trace("TiePointGeoCoding: Max.error X = " + maxErrorX + ", " + (maxErrorX < 0.5 ? "OK" : "too large"));
        Debug.trace("TiePointGeoCoding: Max.error Y = " + maxErrorY + ", " + (maxErrorY < 0.5 ? "OK" : "too large"));
        return new Approximation(fX, fY, centerLat, centerLon, maxSquareDistance * 1.1f);
    }

    private static float getMaxSquareDistance(double[][] data, float centerLat, float centerLon) {
        float maxSquareDistance = 0.0f;
        for (double[] point : data) {
            float dLat = (float)point[0] - centerLat;
            float dLon = (float)point[1] - centerLon;
            float squareDistance = dLat * dLat + dLon * dLon;
            if (!(squareDistance > maxSquareDistance)) continue;
            maxSquareDistance = squareDistance;
        }
        return maxSquareDistance;
    }

    private static Approximation getBestApproximation(Approximation[] approximations, float lat, float lon) {
        Approximation approximation = null;
        if (approximations.length == 1) {
            Approximation a = approximations[0];
            float squareDistance = a.getSquareDistance(lat, lon);
            if (squareDistance < a.getMinSquareDistance()) {
                approximation = a;
            }
        } else {
            float minSquareDistance = Float.MAX_VALUE;
            for (Approximation a : approximations) {
                float squareDistance = a.getSquareDistance(lat, lon);
                if (!(squareDistance < minSquareDistance) || !(squareDistance < a.getMinSquareDistance())) continue;
                minSquareDistance = squareDistance;
                approximation = a;
            }
        }
        return approximation;
    }

    private Approximation findRenormalizedApproximation(float lat, float renormalizedLon, float distance) {
        float renormalizedDistance;
        Approximation renormalizedApproximation = TiePointGeoCoding.getBestApproximation(this.approximations, lat, renormalizedLon);
        if (renormalizedApproximation != null && (renormalizedDistance = renormalizedApproximation.getSquareDistance(lat, renormalizedLon)) < distance) {
            return renormalizedApproximation;
        }
        return null;
    }

    float getNormalizedLonMin() {
        return this.normalizedLonMin;
    }

    private static double rescaleLongitude(double lon, double centerLon) {
        return (lon - centerLon) / 90.0;
    }

    private static double rescaleLatitude(double lat) {
        return lat / 90.0;
    }

    @Override
    public boolean transferGeoCoding(Scene srcScene, Scene destScene, ProductSubsetDef subsetDef) {
        TiePointGrid lonGrid;
        String latGridName = this.getLatGrid().getName();
        String lonGridName = this.getLonGrid().getName();
        Product destProduct = destScene.getProduct();
        TiePointGrid latGrid = destProduct.getTiePointGrid(latGridName);
        if (latGrid == null) {
            latGrid = TiePointGrid.createSubset(this.getLatGrid(), subsetDef);
            destProduct.addTiePointGrid(latGrid);
        }
        if ((lonGrid = destProduct.getTiePointGrid(lonGridName)) == null) {
            lonGrid = TiePointGrid.createSubset(this.getLonGrid(), subsetDef);
            destProduct.addTiePointGrid(lonGrid);
        }
        if (latGrid != null && lonGrid != null) {
            destScene.setGeoCoding(new TiePointGeoCoding(latGrid, lonGrid, this.getDatum()));
            return true;
        }
        return false;
    }

    public static final class Approximation {
        private final FXYSum _fX;
        private final FXYSum _fY;
        private final float _centerLat;
        private final float _centerLon;
        private final float _minSquareDistance;

        public Approximation(FXYSum fX, FXYSum fY, float centerLat, float centerLon, float minSquareDistance) {
            this._fX = fX;
            this._fY = fY;
            this._centerLat = centerLat;
            this._centerLon = centerLon;
            this._minSquareDistance = minSquareDistance;
        }

        public final FXYSum getFX() {
            return this._fX;
        }

        public final FXYSum getFY() {
            return this._fY;
        }

        public float getCenterLat() {
            return this._centerLat;
        }

        public float getCenterLon() {
            return this._centerLon;
        }

        public float getMinSquareDistance() {
            return this._minSquareDistance;
        }

        public final float getSquareDistance(float lat, float lon) {
            float dx = lon - this._centerLon;
            float dy = lat - this._centerLat;
            return dx * dx + dy * dy;
        }
    }
}

